/*
 * Function library for ILI9341 display and XPT2046 touch
 * Designed for Micromite
 * 
 * Uses map and constrain from Arduino https://github.com/arduino/Arduino (GPLV2)
 *  */

#include <xc.h>
#include <stdbool.h>
#include <string.h>
#include "font_arial.h"

#ifndef LCD_BACKPACK_H    /* Guard against multiple inclusion */
#define LCD_BACKPACK_H

//matrix solution
/*
 u,v is raw touch
 x,y is actual coordinates in rotation 2
 x = ua+vb+c
 y = ud+ve+f
 work with ~16 bits fixed point
 */
//for ILI9341
/*
#define Z_TOUCH_THRESHOLD 100
#define TOUCH_A 11612
#define TOUCH_B 0
#define TOUCH_C (-1056568L)
#define TOUCH_D 0
#define TOUCH_E 8411
#define TOUCH_F (-761160L)
#define TOUCH_OVERSAMPLE 16
*/
//for ILI9488
#define Z_TOUCH_THRESHOLD 20
#define TOUCH_A (-16618L)
#define TOUCH_B 0
#define TOUCH_C (33372468L)
#define TOUCH_D 0
#define TOUCH_E (-11050L)
#define TOUCH_F (21811264L)
#define TOUCH_OVERSAMPLE 16

//24 bit colours for ILI9488
#define BLACK 0x0000000
#define BLUE 0x0000FF
#define RED 0xFF0000
#define GREEN 0x00FF00
#define CYAN 0x00FFFF
#define MAGENTA 0xFF00FF
#define YELLOW 0xFFFF00
#define WHITE 0xFFFFFF
#define GREY 0x808080
//unit colours - multiply by 0-31
#define RED_1 0x080000
#define GREEN_1 0x000800
#define BLUE_1 0x000008

/*
//some nice colour names
#define BLACK 0x0000
#define BLUE 0x001F
#define RED 0xF800
#define GREEN 0x07E0
#define CYAN 0x07FF
#define MAGENTA 0xF81F
#define YELLOW 0xFFE0  
#define WHITE 0xFFFF
#define GREY 0x8410
//unit colours - multiply by 0-31
#define RED_1 0x0800
#define GREEN_1 0x0040
#define BLUE_1 0x0001
*/

//pin designations
//tried doing macros, but no luck
//LCD backpack SPI is MOSI:3(RA1), MISO:14(RB5), SCK: 25(RB14) MOSI/MISO are remappable
//LCD backpack is CS on 6 (RB2), DC on 2 (RA0), TOUCHCS on 7 (RB3)
//LCD backlight on 26 (RB15)
//these are used often enough
#define BACKPACK_CS LATBbits.LATB2
#define BACKPACK_DC LATAbits.LATA0
#define BACKPACK_RST LATBbits.LATB12
#define BACKPACK_TOUCHCS LATBbits.LATB3
#define BACKPACK_BACKLIGHT LATBbits.LATB15

#define BUTTON_RADIUS 5
#define BUTTON_FORE WHITE
#define BUTTON_BACK BLACK
#define BUTTON_OTHER CYAN
#define BUTTON_ERROR RED

#define BUTTON_FONT ARIAL

struct backpackbutton{      //core button features, colour, radius etc set by style
    int x0;
    int y0;
    int x1;
    int y1;
    const char *text;
};


int backpackwidth,backpackheight,backpackrotation;

void delay(uint32_t t);
void backpackinit();
void backpackSPIinit();
void backpackrotate(char a);
void backpackdata(unsigned char d);
void backpackcmd(unsigned char d);
unsigned int backpackSPI16(unsigned int n);
void backpackareaset(int x1,int y1,int x2,int y2);    
void backpackbox(int x1,int y1,int x2,int y2,unsigned int c);  
void backpackclear(unsigned int c);
void backpackpoint(int x,int y, unsigned int c);
void backpackline(int x1,int y1,int x2,int y2, unsigned int c);
void backpackhline(int x1,int y1,int x2,unsigned int c);
void backpackvline(int x1,int y1,int y2,unsigned int c);
int backpacktouchxraw();  //raw adc
int backpacktouchyraw();
int backpacktouchzraw();
int backpackscaletouchx();  //scaled to suit rotation 2
int backpackscaletouchy();
int backpacktouchx();     //scaled, rotated, pixel reference
int backpacktouchy();
int backpackcharfont(int x, int y, char c, unsigned int f, unsigned int b, const char* font);
void backpackchararrayfont(int x0, int y0, const char *c, unsigned int f, unsigned int b, const char* font);
long map(long x, long in_min, long in_max, long out_min, long out_max);
void backpackfcircle(int xo,int yo,int r,unsigned int c);
void backpackcircle(int xo,int yo,int r,unsigned int c);
void backpacktriangle(int x1,int y1,int x2,int y2,int x3,int y3,unsigned int c);        //filled triangle
void backpackarc(int xo,int yo,int r,int mask, unsigned int c);     //arc- mask is for octants, starting at 12-1:30 = bit 0
void backpackrbox(int x1,int y1,int x2,int y2,int r, unsigned int c); //box with rounded corners
void backpackquad(int xo,int yo,int r,int mask, unsigned int c);     //quadrant, filled- mask is, starting at 12-3 = bit 0
void backpacktextbutton(int x1,int y1,int x2,int y2,int r, const char *c,unsigned int f,unsigned int b,const char* font); //micromite style button
void backpacksetbuttonobject(struct backpackbutton *b,int x0,int y0,int x1,int y1, const char *text);      //set up button object
bool backpackbuttonobjecttouched(struct backpackbutton *b);
void backpackdrawbuttonobject(struct backpackbutton *b, bool pressed, int r, unsigned int f,unsigned int bg,const char* font);
int backpackcheckbuttons(struct backpackbutton *b, int start, int end);
void buttonStyle(struct backpackbutton *b, bool pressed);
void buttonErase(struct backpackbutton *b, bool pressed);
void backpackmonobitmap(int x, int y, const uint8_t* bm);
void backpackmonobitmapscale(int x, int y, const uint8_t* bm,int s);
void backpacknint(int x, int y, unsigned int n,int d, unsigned int f,unsigned int bg,const char* font);
void backpackshowms(int x, int y, unsigned int n, unsigned int f,unsigned int bg,const char* font);  //show time in m:s to d digits)    

#endif /* LCD_BACKPACK_H */
